<?php

namespace App\Http\Controllers;

use App\Models\Clients;
use App\Models\ModePaiment;
use App\Models\Products;
use App\Models\Settings;
use App\Models\StatementItems;
use App\Models\Statements;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class StatementsController extends Controller
{
    public function invoices()
    {
        $clients = Clients::where('type', 0)->get();
        $query = Statements::with('client', 'reglements')->where('type', 1);



        if (request()->has('date_debut') && request()->has('date_fin')) {
            $date_debut = request()->date_debut;
            $date_fin = request()->date_fin;
            if ($date_debut > $date_fin) {
                return redirect()->back()->withErrors(['date' => 'La date de début doit être inférieure à la date de fin']);
            }
            $query->whereBetween('date', [$date_debut, $date_fin]);
        } else if (request()->has('date_debut')) {
            $date_debut = request()->date_debut;
            $query->where('date', '=', $date_debut); // Include statements on the exact date
        }

        $statements = $query->orderBy('created_at', 'desc')->get();
        if (request()->has('client')) {
            $client = request()->client;
            $statements = $statements->filter(function ($statement) use ($client) {
                return $statement->client_id == $client;
            });
        }
        return view('pages.statements.index', ['type' => 1, 'statements' => $statements, 'clients' => $clients]);
    }

    public function statements()
    {
        $statements = Statements::with('client')->where('type', 2)->orderBy('created_at', 'desc')->get();
        return view('pages.statements.index', ['type' => 2, 'statements' => $statements]);
    }

    public function bons()
    {
        $statements = Statements::with('client', 'statement')->where('type', 3)->orderBy('created_at', 'desc')->get();
        return view('pages.statements.index', ['type' => 3, 'statements' => $statements]);
    }

    public function show_invoice($id = null)
    {
        $statement = null;
        if ($id) {
            $statement = Statements::with('items', 'user', 'reglements')->findOrFail($id);
        }
        $products = Products::with('bon_d_entrees', 'items', 'items.statement', 'items.statement.reglements')->get();
        $clients = Clients::where('type', 0)->get();
        $timbre = Settings::first()->tmbr;
        $modes = ModePaiment::all();

        $newestBon = Statements::where('type', 1)->orderBy('created_at', 'desc')->first();
        $ref = null;
        if (!$newestBon) {
            $ref = 'FACT-' . date('Y') . '-0001';
        } else {
            $ref = $newestBon->getId();
        }
        $date = now()->toDateString();
        return view('pages.statements.show', [
            'type' => 1,
            'statement' => $statement,
            'products' => $products,
            'clients' => $clients,
            'timbre' => $timbre,
            'modes' => $modes,
            'ref' => $ref,
            'date' => $date
        ]);
    }

    public function show_statement($id = null)
    {
        $statement = null;
        if ($id) {
            $statement = Statements::with('items', 'user')->findOrFail($id);
        }
        $products = Products::with('bon_d_entrees')->get();
        $clients = Clients::where('type', 0)->get();
        $timbre = Settings::first()->tmbr;
        $modes = ModePaiment::all();
        $newestBon = Statements::where('type', 2)->orderBy('created_at', 'desc')->first();
        $ref = null;
        if (!$newestBon) {
            $ref = 'DEV-' . date('Y') . '-0001';
        } else {
            $ref = $newestBon->getId();
        }
        $date = now()->toDateString();
        return view('pages.statements.show', ['type' => 2, 'statement' => $statement, 'products' => $products, 'clients' => $clients, 'timbre' => $timbre, 'modes' => $modes, 'ref' => $ref, 'date' => $date]);
    }

    public function show_bon($id = null)
    {
        $statement = null;
        if ($id) {
            $statement = Statements::with('items', 'user')->findOrFail($id);
        }
        $products = Products::with('bon_d_entrees')->get();
        $clients = Clients::where('type', 0)->get();
        $timbre = Settings::first()->tmbr;
        $modes = ModePaiment::all();
        $newestBon = Statements::where('type', 3)->orderBy('created_at', 'desc')->first();
        $ref = null;
        if (!$newestBon) {
            $ref = 'BL' . date('Y') . '0001';
        } else {
            $ref = $newestBon->getId();
        }
        $date = now()->toDateString();
        return view('pages.statements.show', ['type' => 3, 'statement' => $statement, 'products' => $products, 'clients' => $clients, 'timbre' => $timbre, 'modes' => $modes, 'ref' => $ref, 'date' => $date]);
    }

    public function store(Request $request)
    {

        $form_fields = $request->validate([
            'client_id' => ['required', 'numeric'],
            'timbreFiscal' => ['nullable', 'numeric'],
            'type' => ['required', 'string'],
            'comment' => ['nullable', 'string'],
            'date' => ['nullable', 'date']
        ]);
        $form_fields['user_id'] = auth()->user()->id;
        $form_fields['is_generated'] = 0;
        $newestStatement = Statements::where('type', $form_fields['type'])->orderBy('created_at', 'desc')->first();
        if (!$newestStatement) {
            $form_fields['reference'] = ($form_fields['type'] == 1) ? 'FA' . date('Y') . '0001' : 'DEV' . date('Y') . '0001';
            if ($form_fields['type'] == 3) {
                $form_fields['reference'] = 'BL' . date('Y') . '0001';
            }
        } else {
            $form_fields['reference'] = $newestStatement->getId();
        }
        $statement = Statements::create($form_fields);
        $items = $request->input('items');
        $items = json_decode($items, true);
        $tht = 0;
        $tva = 0;
        $gqte = 0;
        $settings = Settings::all()->first();

        foreach ($items as $item) {
            $statementItem = new StatementItems();
            $statementItem->statement_id = $statement->id;
            $product = Products::with('bon_d_entrees')->where('id', $item['id'])->first();
            $quantity = 0;
            foreach ($product->bon_d_entrees as $bon) {
                $quantity += $bon->pivot->quantity;
            }
            if (($quantity + $product->quantity < $item['quantité']) && $settings->qtezero == 1 && $statement->type == 1) {
                $statement->delete();
                return redirect()->back()->withErrors(['quantité' => 'La quantité demandée est supérieure à la quantité en stock']);
            }
            $statementItem->name = $item['name'];
            $statementItem->code = $item['code'];
            $statementItem->product_id = $item['id'];
            $statementItem->quantity = $item['quantité'];
            $statementItem->price = $item['price'];
            $statementItem->tva = $item['tva'];
            $statementItem->discount = $item['discount'];
            $gqte = $gqte + $statementItem->quantity;
            $price = ($statementItem->price * $statementItem->quantity) * (100 - $statementItem->discount) / 100;
            $tht = $tht + $price;
            $tva = $tva + ($statementItem->tva * $price) / 100;
            $statementItem->save();
            $product->quantity -= $item['quantité'];
            $product->save();
        }

        $statement->tht = $tht;
        $statement->tva = $tva;
        $statement->ttc = $tht + $tva;
        $statement->gqte = $gqte;
        if ($request->type == 1) {
            $statement->ttc = $statement->ttc + $statement->timbreFiscal;
        }
        $statement->update();
        if ($statement->type == 1) {
            return redirect()->route('invoices.index');
        }
        if ($statement->type == 3) {
            return redirect()->route('bons.index');
        }
        return redirect()->route('statements.index');
    }

    public function update(Request $request, $id)
    {
        $form_fields = $request->validate([
            'client_id' => ['required', 'numeric'],
            'timbreFiscal' => ['nullable', 'numeric'],
            'type' => ['required', 'string'],
            'comment' => ['nullable', 'string'],
            'date' => ['nullable', 'date']
        ]);
        $statement = Statements::with('items')->findOrFail($id);
        $statement->edit_id = auth()->user()->id;
        $statement->update($form_fields);
        $oldItems = $statement->items;
        $statement->items()->delete();
        $items = $request->input('items');
        $items = json_decode($items, true);
        $tht = 0;
        $tva = 0;
        $gqte = 0;
        $settings = Settings::all()->first();
        foreach ($items as $item) {
            $statementItem = new StatementItems();
            $statementItem->statement_id = $statement->id;
            $product = Products::with('bon_d_entrees')->where('id', $item['id'])->first();
            $quantity = 0;
            foreach ($product->bon_d_entrees as $bon) {
                $quantity += $bon->pivot->quantity;
            }
            if (($quantity + $product->quantity < $item['quantité']) && $settings->qtezero == 1 && $statement->type == 1) {

                return redirect()->back()->withErrors(['quantité' => 'La quantité demandée est supérieure à la quantité en stock']);
            }
            $oldQuantity = $oldItems->firstWhere('product_id', $item['id'])->quantity ?? 0;
            $statementItem->name = $item['name'];
            $statementItem->code = $item['code'];
            $statementItem->product_id = $item['id'];
            $statementItem->quantity = $item['quantité'];
            $statementItem->price = $item['price'];
            $statementItem->tva = $item['tva'];
            $statementItem->discount = $item['discount'];
            $gqte = $gqte + $statementItem->quantity;
            $price = ($statementItem->price * $statementItem->quantity) * (100 - $statementItem->discount) / 100;
            $tht = $tht + $price;
            $tva = $tva + ($statementItem->tva * $price) / 100;
            $statementItem->save();
            $product->quantity += $oldQuantity;
            $product->quantity -= $item['quantité'];
            $product->save();
        }

        $statement->tht = $tht;
        $statement->tva = $tva;
        $statement->ttc = $tht + $tva;
        $statement->gqte = $gqte;
        if ($request->type == 1) {
            $statement->ttc = $statement->ttc + $statement->timbreFiscal;
        }
        if ($statement->type == 3) {
            $statement->is_generated = 0;
        }
        $statement->update();
        if ($statement->type == 1) {
            return redirect()->route('invoices.index');
        }
        if ($statement->type == 3) {
            return redirect()->route('bons.index');
        }
        return redirect()->route('statements.index');
    }

    public function destroy($type)
    {
        $statement = Statements::with('items')->where('type', $type)->latest()->first();
        if ($statement != null) {

            foreach ($statement->items as $item) {
                $product = Products::with('bon_d_entrees')->where('id', $item->product_id)->first();
                $product->quantity += $item->quantity;
                $product->save();
                
            }

            $statement->items()->delete();
            $statement->reglements()->delete();
            if ($statement->statement != null) {
                foreach ($statement->statement as $st) {
                    $st->is_generated = 0;
                    $st->statement_id = null;
                    $st->update();
                }
            }
            $statement->delete();
        }
        if ($type == 1) {
            return redirect()->back()->with('success', 'Devis supprimé avec succès');
        }
        return redirect()->back()->with('success', 'Facture supprimée avec succès');
    }

    public function toInvoice($id)
    {
        $statement = Statements::findOrFail($id);

        $invoice = new Statements();
        $invoice->client_id = $statement->client_id;
        $invoice->timbreFiscal = $statement->timbreFiscal;
        $invoice->tht = $statement->tht;
        $invoice->tva = $statement->tva;
        $invoice->date = now();
        $invoice->gqte = $statement->gqte;
        $invoice->user_id = auth()->user()->id;
        $invoice->edit_id = auth()->user()->id;
        $invoice->comment = $statement->comment;
        $invoice->is_generated = 0;
        $invoice->type = 1;
        $settings = Settings::all()->first();
        $invoice->timbreFiscal = $settings->tmbr;
        $invoice->ttc = $statement->ttc + $invoice->timbreFiscal;
        $newestStatement = Statements::where('type', 1)->orderBy('created_at', 'desc')->first();

        if (!$newestStatement) {
            $invoice->reference =  'FA' . date('Y') . '0001';
        } else {
            $invoice->reference = $newestStatement->getId();
        }
        $invoice->save();
        $statementItems = StatementItems::where('statement_id', $statement->id);
        foreach ($statementItems->get() as $item) {
            $newItem = new StatementItems();
            $newItem->statement_id = $invoice->id;
            $newItem->product_id = $item->product_id;
            $newItem->name = $item->name;
            $newItem->code = $item->code;
            $newItem->quantity = $item->quantity;
            $product = Products::with('bon_d_entrees')->where('id', $item->product_id)->first();
            $quantity = 0;
            foreach ($product->bon_d_entrees as $bon) {
                $quantity += $bon->pivot->quantity;
            }
            if (($quantity + $product->quantity < $item['quantité']) && $settings->qtezero == 1) {
                $invoice->delete();
                return redirect()->back()->withErrors(['quantité' => 'La quantité demandée est supérieure à la quantité en stock']);
            }
            $newItem->price = $item->price;
            $newItem->discount = $item->discount;
            $newItem->tva = $item->tva;
            $newItem->save();
           
            $product->quantity -= $item['quantité'];
            $product->save();
        }

        if ($statement->type == 3) {
            $statement->is_generated = 1;
            $statement->statement_id = $invoice->id;
            $statement->update();
        }

        return redirect()->route('invoices.index');
    }

    public function toBon($id)

    {
        $statement = Statements::findOrFail($id);

        $invoice = new Statements();
        $invoice->client_id = $statement->client_id;
        $invoice->timbreFiscal = $statement->timbreFiscal;
        $invoice->tht = $statement->tht;
        $invoice->tva = $statement->tva;
        $invoice->gqte = $statement->gqte;
        $invoice->user_id = auth()->user()->id;
        $invoice->edit_id = auth()->user()->id;
        $invoice->comment = $statement->comment;
        $invoice->date = $statement->date;
        $invoice->type = 3;
        $invoice->is_generated = 1;
        $invoice->statement_id = $statement->id;
        $settings = Settings::all()->first();
        $invoice->timbreFiscal = $settings->tmbr;
        $invoice->ttc = $statement->ttc + $invoice->timbreFiscal;
        $newestStatement = Statements::where('type', 1)->orderBy('created_at', 'desc')->first();
        if (!$newestStatement) {
            $invoice->reference =  'BL' . date('Y') . '0001';
        } else {
            $invoice->reference = $newestStatement->getId();
        }
        $invoice->save();
        $statementItems = StatementItems::where('statement_id', $statement->id);
        foreach ($statementItems->get() as $item) {
            $newItem = new StatementItems();
            $newItem->statement_id = $invoice->id;
            $newItem->product_id = $item->product_id;
            $newItem->name = $item->name;
            $newItem->code = $item->code;
            $newItem->quantity = $item->quantity;
            $product = Products::with('bon_d_entrees')->where('id', $item->product_id)->first();
            $quantity = 0;
            foreach ($product->bon_d_entrees as $bon) {
                $quantity += $bon->pivot->quantity;
            }
            if (($quantity + $product->quantity < $item['quantité']) && $settings->qtezero == 1) {
                $invoice->delete();
                return redirect()->back()->withErrors(['quantité' => 'La quantité demandée est supérieure à la quantité en stock']);
            }
            $newItem->price = $item->price;
            $newItem->discount = $item->discount;
            $newItem->tva = $item->tva;
            $newItem->save();
        }

        return redirect()->route('bons.index');
    }


    public function getPDF(Statements $statement)
    {
        $filename = $statement->type . '-' . $statement->reference . '.pdf';
        $date = date('d/m/Y', strtotime($statement->updated_at));
        $settings = Settings::find(1)->get();
        $statementItems = StatementItems::where("statement_id", $statement->id)->get();
        foreach ($statementItems as $item) {
            $product = Products::where('id', $item->product_id)->first();
            $item->product = $product->name;
            $item->image = $product->image;
            $item->code = $product->code;
        }
        $client = Clients::where('id', $statement->client_id)->first();
        $settings = Settings::all()->first();
        $data = [
            "name" => $filename,
            "statement" => $statement,
            "items" => $statementItems,
            "company" => $settings,
            "date" => $date,
            "client" => $client,
            "tvaCollection" => $this->tvaCollector($statement),
            "totalDiscount" => $this->totalDiscount($statement),
        ];
        $pdf = \PDF::loadView('pdf.statement', $data);
        $pdf->render();

        $dompdf = $pdf->getDomPDF();
        $canvas = $dompdf->get_canvas();
        $font = $dompdf->getFontMetrics()->get_font("helvetica", "semibold");

        // Get the canvas dimensions
        $width = $canvas->get_width();
        $height = $canvas->get_height();

        // Set Y-coordinate for footer content
        $footerY = $height - 25;

        $line = "____________________________________________________________________________________________________________";
        $canvas->page_text(20, $footerY - 12, $line, $font, 9, array(0, 0, 0)); // Left-aligned date
        // Footer content
        $canvas->page_text(20, $footerY, $date, $font, 9, array(0, 0, 0)); // Left-aligned date
        $canvas->page_text($width / 4, $footerY, "RIB: {$settings->rib} {$settings->bank}, {$settings->agency}", $font, 9, array(0, 0, 0), 'center'); // Centered RIB

        // Page numbers (right-aligned)
        $canvas->page_text($width - 90, $footerY, "Page {PAGE_NUM} sur {PAGE_COUNT}", $font, 9, array(0, 0, 0));

        return $pdf->stream($filename);
    }

    public function tvaCollector(Statements $statement)
    {
        $statementItems = StatementItems::where("statement_id", $statement->id)->get();
        $tvaCollection = [];
        foreach ($statementItems as $item) {

            $base = ($item->price * $item->quantity) * (100 - $item->discount) / 100;
            $amount = $base * ($item->tva) / 100;
            $key = (string) $item->tva;
            if (array_key_exists($key, $tvaCollection)) {
                $tvaCollection[$key][0] =  $tvaCollection[$key][0] + $base;
                $tvaCollection[$key][1] =  $tvaCollection[$key][1] + $amount;
            } else {
                $tvaCollection[$key][0] =  $base;
                $tvaCollection[$key][1] =  $amount;
            }
        }
        return $tvaCollection;
    }

    public function totalDiscount(Statements $statement)
    {
        $statementItems = StatementItems::where("statement_id", $statement->id)->get();
        $totalDiscount = 0;
        foreach ($statementItems as $item) {
            $totalDiscount = $totalDiscount + ($item->discount * $item->price * $item->quantity) / 100;
        }
        return $totalDiscount;
    }

    public function getBonDeLivraison(Statements $statement)
    {
        $filename = $statement->type . '-' . str_replace('FA', 'BL', $statement->reference) . '.pdf';
        $date = date('d/m/Y', strtotime($statement->updated_at));
        $settings = Settings::find(1)->get();
        $statementItems = StatementItems::where("statement_id", $statement->id)->get();
        foreach ($statementItems as $item) {
            $product = Products::where('id', $item->product_id)->first();
            $item->product = $product->name;
            $item->image = $product->image;
            $item->code = $product->code;
        }
        $client = Clients::where('id', $statement->client_id)->first();
        $settings = Settings::all()->first();
        $data = [
            "name" => $filename,
            "statement" => $statement,
            "items" => $statementItems,
            "company" => $settings,
            "date" => $date,
            "client" => $client,
        ];
        $pdf = \PDF::loadView('pdf.livraison', $data);
        $pdf->render();

        $dompdf = $pdf->getDomPDF();
        $canvas = $dompdf->get_canvas();
        $font = $dompdf->getFontMetrics()->get_font("helvetica", "semibold");

        // Get the canvas dimensions
        $width = $canvas->get_width();
        $height = $canvas->get_height();

        // Set Y-coordinate for footer content
        $footerY = $height - 25;

        $line = "____________________________________________________________________________________________________________";
        $canvas->page_text(20, $footerY - 12, $line, $font, 9, array(0, 0, 0)); // Left-aligned date
        // Footer content
        $canvas->page_text(20, $footerY, $date, $font, 9, array(0, 0, 0)); // Left-aligned date
        $canvas->page_text($width / 4, $footerY, "RIB: {$settings->rib} {$settings->bank}, {$settings->agency}", $font, 9, array(0, 0, 0), 'center'); // Centered RIB

        // Page numbers (right-aligned)
        $canvas->page_text($width - 90, $footerY, "Page {PAGE_NUM} sur {PAGE_COUNT}", $font, 9, array(0, 0, 0));

        return $pdf->stream($filename);
    }

    public function generateInvoiceFromMultiStatements(Request $request)
    {
        $statements = $request->input('ids');
        $previousClientId = null;
        if (gettype($statements) == 'string') {
            $statements = json_decode($statements);
        }
        $invoices = [];
        foreach ($statements as $statement) {
            $invoice = Statements::findOrFail($statement);
            if ($previousClientId !== null && $invoice->client_id !== $previousClientId) {
                return redirect()->back()->withErrors(['error' => 'Vous ne pouvez pas générer une facture pour des clients différents']);
            }
            array_push($invoices, $invoice);
        }
        if (count($invoices) == 0) {
            return redirect()->back()->withErrors(['error' => 'Aucun élément sélectionné']);
        }

        $newInvoice = new Statements();
        $newInvoice->client_id = $invoices[0]->client_id;
        $settings = Settings::all()->first();
        $newInvoice->timbreFiscal = $settings->tmbr;
        $newInvoice->user_id = auth()->user()->id;
        $newInvoice->edit_id = auth()->user()->id;
        $newInvoice->type = 1;
        $newInvoice->date = now();
        $newestStatement = Statements::where('type', '1')->orderBy('created_at', 'desc')->first();
        if ($newestStatement == null) {
            $newInvoice->reference =  'FA' . date('Y') . '0001';
        } else {
            $newInvoice->reference = $newestStatement->getId();
        }
        $newInvoice->save();

        foreach ($invoices as $invoice) {
            $invoice->comment ? $newInvoice->comment = $newInvoice->comment . PHP_EOL . ' ' . $invoice->comment : null;
            $statementItems = StatementItems::where('statement_id', $invoice->id)->get();
            foreach ($statementItems as $item) {
                $newItem = new StatementItems();
                $newItem->statement_id = $newInvoice->id;
                $newItem->product_id = $item->product_id;
                $newItem->name = $item->name;
                $newItem->code = $item->code;
                $newItem->quantity = $item->quantity;
                $product = Products::with('bon_d_entrees')->where('id', $item->product_id)->first();
                $quantity = 0;
                foreach ($product->bon_d_entrees as $bon) {
                    $quantity += $bon->pivot->quantity;
                }
                if (($quantity + $product->quantity < $item['quantité']) && $settings->qtezero == 1) {
                    $invoice->delete();
                    return redirect()->back()->withErrors(['quantité' => 'La quantité demandée est supérieure à la quantité en stock']);
                }
                $newItem->price = $item->price;
                $newItem->discount = $item->discount;
                $newItem->tva = $item->tva;
                $newItem->save();
                

                $product->quantity -= $item['quantité'];
                $product->save();
            }
            $newInvoice->tht += $invoice->tht;
            $newInvoice->tva += $invoice->tva;
            $newInvoice->gqte += $invoice->gqte;

            if ($invoice->type == 3) {
                $invoice->is_generated = 1;
                $invoice->statement_id = $newInvoice->id;
                $invoice->update();
            }
        }
        $newInvoice->ttc = $newInvoice->tht + $newInvoice->tva + $newInvoice->timbreFiscal;
        $newInvoice->is_generated = 0;
        $newInvoice->update();
        return $newInvoice->id;
    }
}
