<?php

namespace App\Http\Controllers;

use App\Models\Products;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class ProductsController extends Controller
{
    public function index(Request $request)
    {

        return view('pages.products.index');
    }

    public function getProducts(Request $request)
    {
        $columns = [
            'name',
            'code',
            'description',
            'price',
            'quantity',
            'tva'
        ];

        $search = $request->input('search');
        $orderColumnIndex = $request->input('order.0.column');
        $orderDirection = $request->input('order.0.dir', 'asc');
        $orderColumn = $columns[$orderColumnIndex] ?? 'name';

        $query = Products::query();


        // Apply search
        if (!empty($search)) {
            $query->where(function ($q) use ($columns, $search) {
                foreach ($columns as $column) {
                    $q->orWhere($column, 'LIKE', "%{$search}%");
                }
            });
        }

        $query->orderBy($orderColumn, $orderDirection);
        $totalRecords = $query->count();

        // Apply pagination
        $start = $request->input('start', 0);
        $length = $request->input('length', 30);
        $query->offset($start)->limit($length);

        $data = $query->get();

        $response = [
            'draw' => $request->input('draw'),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ];

        return response()->json($response);
    }

    public function store(Products $product)
    {
        $form_fields = request()->validate([
            'name' => 'required',
            'code' => 'nullable',
            'description' => 'nullable',
            'price' => 'nullable',
            'tva' => 'nullable',
        ]);
        if (request()->hasFile('image')) {
            $file = request()->file('image');
            $fileName =  $file->getClientOriginalName();
            if ($file->move(public_path() . '/storage/products/', $fileName)) {
                $form_fields['image'] = $fileName;
            }
        }
        $form_fields['quantity'] = 0;
        $product = Products::create($form_fields);
        return redirect()->back();
    }

    public function show($id)
    {
        $product = Products::findOrFail($id);
        return response()->json($product);
    }

    public function update($id, Request $request)
    {
        $product = Products::findOrFail($id);
        $form_fields = request()->validate([
            'name' => 'required',
            'code' => 'nullable',
            'description' => 'nullable',
            'price' => 'nullable',
            'tva' => 'nullable',
        ]);
        if (request()->hasFile('image') && isset($product->image) && !str_starts_with($product->image, 'http')) {
            $file = request()->file('image');
            $fileName =  $file->getClientOriginalName();
            if ($file->move(public_path() . '/storage/products/', $fileName)) {
                $form_fields['image'] = $fileName;
            }
        } else {
            $form_fields['image'] = null;
        }
        $product->update($form_fields);
        return redirect()->back();
    }

    public function destroy($id)
    {
        $product = Products::with('bon_d_entrees', 'items')->findOrFail($id);
        foreach ($product->items as $item) {
            $item->delete();
        }
        foreach ($product->bon_d_entrees as $bon) {
            $bon->delete();
        }
        $product->delete();
        return redirect()->back();
    }

    public function synchronize()
    {
        $url = 'https://fk-info.com/new.php';
        $response = Http::get($url);
        $response = $response->json();

        try {
            $processedCount = 0;
            $totalCount = count($response); // Total number of products in the response

            foreach ($response as $product) {
                $product = (array) $product;
                $updatedProduct = Products::where('code', $product['code'])->first();

                if ($updatedProduct) {
                    // Update existing product
                    $updatedProduct->update($product);
                } else {
                    // Create new product
                    Products::create($product);
                }

                $processedCount++; // Increment count for each product processed
            }

            // Return success response with both processed and total counts
            return response()->json([
                'message' => 'Synchronization completed successfully',
                'processed_count' => $processedCount,
                'total_count' => $totalCount
            ], 200);
        } catch (Exception $e) {
            // Return error response with 500 status code
            return response()->json(['message' => 'Internal Server Error'], 500);
        }
    }
}
